// fixme: compile ranges for pass1 and pass2
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdbool.h>

#include "opcodes.h"

FILE *inFile;
FILE *outFile;

uint16_t usage[0xff];   // is this opcode used? 256 opcodes in total

bool labels[0x10000];   // label used flags, 64kb (65536) max code size

char stringBuffer[1024];

void decompile(uint8_t data[], uint16_t progCnt);

uint16_t relativeAddress(uint16_t progCount, uint8_t opcode0, uint8_t relAddr);
void partialDecompile(uint8_t data[], uint16_t progCnt);

void pass1();
void pass2();

void pass1() {
  for (uint16_t i = 0; i<=0xfff; i++) {
    labels[i] = false;
  }
  uint16_t PC = 0;

  while ( true ) {
    uint8_t opcodeArray[3];
    uint16_t PCpass = 0;
    uint8_t retVal;
    uint8_t opCode;
    size_t err;

    // firmware file size
    if (PC > 0xFFF) break;

    err = fread(&opCode, 1, 1, inFile);
    if (err != 1) {
      printf("Error reading from source in while loop pass1 opcode[0]!\n");
      exit(-1);
    }

    PCpass = PC;
    opcodeArray[0] = opCode;

    // do not decode because string is here
    if (PC >=0xA40 && PC <=0xA75 ) {
      PC++;
      continue;
    }
    
    // no code after this address, do nothing
    if (PC <= 0xB5A) {
      for (uint8_t i = 1; i < opcodeLen[opCode]; i++ ) {
        err = fread(&retVal, 1, 1, inFile);
        if (err != 1) {
          printf("Error reading from source in while loop pass1 opcodes[]!\n");
          exit(-1);
        }
        opcodeArray[i] = retVal;
        PC++;
      }
      partialDecompile(opcodeArray, PCpass);
    }
    PC++;
  }
}

void pass2() {

  fprintf(outFile, "#include <stdio.h>\n");
  fprintf(outFile, "#include <stdlib.h>\n");
  fprintf(outFile, "#include <stdint.h>\n");
  fprintf(outFile, "#include <stdbool.h>\n");
  fprintf(outFile, "\n");
  fprintf(outFile, "// nice routines from friend\n");
  fprintf(outFile, "#define bitRead(value, bit) (((value) >> (bit)) & 0x01)\n");
  fprintf(outFile, "#define bitSet(value, bit) ((value) |= (1UL << (bit)))\n");
  fprintf(outFile, "#define bitClear(value, bit) ((value) &= ~(1UL << (bit)))\n");
  fprintf(outFile, "#define bitToggle(value, bit) ((value) ^= (1UL << (bit)))\n");
  fprintf(outFile, "#define bitWrite(value, bit, bitvalue) ((bitvalue) ? bitSet(value, bit) : bitClear(value, bit))\n");
  fprintf(outFile, "\n");

  fprintf(outFile, "#define bit0 0\n");
  fprintf(outFile, "#define bit1 1\n");
  fprintf(outFile, "#define bit2 2\n");
  fprintf(outFile, "#define bit3 3\n");
  fprintf(outFile, "#define bit4 4\n");
  fprintf(outFile, "#define bit5 5\n");
  fprintf(outFile, "#define bit6 6\n");
  fprintf(outFile, "#define bit7 7\n");
  fprintf(outFile, "\n");
  
  fprintf(outFile, "uint8_t Port0, Port1, Port2, Port3;\n");
  fprintf(outFile, "uint8_t TCON, SCON, IntEnable, IntPriority, PSW;\n");
  
  fprintf(outFile, "\n");

  fprintf(outFile, "uint8_t RAM[0x100];\n");
  fprintf(outFile, "uint8_t accumulator, regB, tmp8;\n");
  fprintf(outFile, "uint8_t R0, R1, R2, R3, R4, R5, R6, R7;\n");
  fprintf(outFile, "uint16_t DPTR, tmp16;\n");
  fprintf(outFile, "bool carry, tmpCarry;\n");
  fprintf(outFile, "\n");

  fprintf(outFile, "int main(int argc, char **argv) {\n");
  fprintf(outFile, "\n");

  uint16_t PC = 0;

  while ( true ) {
    uint8_t opcodeArray[3];   // up to 3 bytes per opcode
    uint16_t PCpass = 0;    // program counter to paas to decompiler
    uint8_t retVal;         // return byte from input file for opcode array
    uint8_t opCode;     // opcode readed from input file (neet to rewrite/remove
    size_t err;         // for file handling

    // firmware file size
    if (PC > 0xFFF) break;

    err = fread(&opCode, 1, 1, inFile);
    if (err != 1) {
      printf("Error reading from source in while loop pass2 opcode[0]!\n");
      exit(-1);
    }
    //printf("label_%03X:\n", PC);

    PCpass = PC;
    opcodeArray[0] = opCode;

    // string, do not pares, write as text or hex
    if (PC >=0xA40 && PC <=0xA75 ) {
      if ( (opcodeArray[0] >= 0x20) && (opcodeArray[0] < 0x7F) ) {
        fprintf(outFile, "%c", opcodeArray[0]);
      } else {
        fprintf(outFile, "0x%02X, ", opcodeArray[0]);
      }
      PC++;
      continue;
    }
    
    // if more, than code ends, so not analyze, or write as text
    if (PC <= 0xB5A) {
      // print label if detected or force to print first 0x25 labels (jump table)
      if ( labels[PC] || PC <= 0x25) {
        fprintf(outFile, "label_%03X:", PC);
      }
      
      // print opcode and bunary. useful for decompiler debug
      // and double check what opcode do
      fprintf(outFile, "\t// addr: %03X\topcodes: %02X", PCpass, opCode);
      // fill array with opcode bytes
      for (uint8_t i = 1; i < opcodeLen[opCode]; i++ ) {
        err = fread(&retVal, 1, 1, inFile);
        if (err != 1) {
          printf("Error reading from source in while loop pass2 opcodes[]!\n");
          exit(-1);
        }
        opcodeArray[i] = retVal;
        fprintf(outFile, " %02X", retVal);
        PC++;
      }
      fprintf(outFile, "\n\t");
      decompile(opcodeArray, PCpass);
    } else {    // no code, write as text
      if ( (opcodeArray[0] >= 0x20) && (opcodeArray[0] < 0x7F) ) {
        fprintf(outFile, "%c", opcodeArray[0]);
      }
    }
    PC++;
  }
  
  fprintf(outFile, ";\n");
  
/*
  printf("unused opcodes:\n");
  for (int i = 0; i <= 0xFF; i++) {
    if (usage[i] == 0 ) {
      printf("opcode: %02X\n", i);
    }
  }
  printf("\n");
*/

/*
  printf("used labels:\n");
  for (uint16_t i = 0; i <= 0xfff; i++) {
    if ( labels[i] ) {
      printf("label_0x%03X\n", i);
    }
  }
  printf("\n");
*/
}

int main(int argc, char **argv) {
  // do pass 1: analyze what labels are used
  // open input binary file for reading
  inFile = fopen(argv[1], "rb");
  if (inFile == NULL) {
    printf("Error opening file %s !\n", argv[1]);
    exit(-1);
  }

  pass1();
  // close input binary file because I am too lazy to use seek
  fclose(inFile);

  // do pass 2: ganerate C code in output file
  // reopen input binary file for reading because I am too lazy to use seek
  inFile = fopen(argv[1], "rb");
  if (inFile == NULL) {
    printf("Error opening file %s !\n", argv[1]);
    exit(-1);
  }

  // open result file for writing
  outFile = fopen(argv[2], "w");
  if (outFile == NULL) {
    printf("Error opening file %s !\n", argv[2]);
    exit(-1);
  }

  pass2();

  fclose(inFile);
  fclose(outFile);

  return 0;
}


// fixme: add relativeAddress() where nacessary
void decompile(uint8_t data[], uint16_t progCnt) {

  uint8_t tmp8;
  uint16_t tmp16;
  int8_t tmp8int;

  usage[data[0]]++;

  switch(data[0]) {
    case 0x00:    // NOP
      fprintf(outFile, "// NOP\n");
      break;
    case 0x01:    // AJMP
    case 0x21:    // AJMP
    case 0x41:    // AJMP
    case 0x61:    // AJMP
    case 0x81:    // AJMP
    case 0xA1:    // AJMP
    case 0xC1:    // AJMP
    case 0xE1:    // AJMP
      tmp16 = ((uint16_t)(data[0] & 0b11100000) << 3) + (uint16_t)data[1];
      tmp16 = ( (progCnt + 1) & 0b1111100000000000) + tmp16;
      //labels[tmp16] = true;
      fprintf(outFile, "goto label_%03X; // absolute jump\n", tmp16);
      break;
    case 0x11:    // ACALL
    case 0x31:    // ACALL
    case 0x51:    // ACALL
    case 0x71:    // ACALL
    case 0x91:    // ACALL
    case 0xB1:    // ACALL
    case 0xD1:    // ACALL
    case 0xF1:    // ACALL
      tmp16 = ((uint16_t)(data[0] & 0b11100000) << 3) + (uint16_t)data[1];
      tmp16 = ( (progCnt+1) & 0b1111100000000000) + tmp16;
      //labels[tmp16] = true;
      fprintf(outFile, "label_%03X(); // absolute call\n", tmp16);
      break;

    case 0x02:    // LJMP
      tmp16 = ((uint16_t)data[1] << 8) + (uint16_t)data[2];
      //labels[tmp16] = true;
      fprintf(outFile, "goto label_%03X; // long jump\n", tmp16);
      break;
    case 0x12:    // LCALL
      tmp16 = ((uint16_t)data[1] << 8) + (uint16_t)data[2];
      //labels[tmp16] = true;
      fprintf(outFile, "label_%03X(); // long call\n", tmp16);
      break;
    case 0x22:    // RET
      fprintf(outFile, "return; // return (opcode: 0x%02X)\n", data[0]);
      break;
    case 0x32:    // RETI
      fprintf(outFile, "return; // return form IRQ (opcode: 0x%02X)\n", data[0]);
      break;

    case 0x03:    // RR A
      fprintf(outFile, "// Waring: RR A Not implemented yet!\n");
      break;
    case 0x04:    // INC A
      fprintf(outFile, "accumulator++;\n");
      break;
    case 0x05:    // INC data addr
      fprintf(outFile, "RAM[0x%02X]++;\n", data[1]);
      break;
    case 0x06:    // INC @R0
      fprintf(outFile, "RAM[R0]++;\n");
      break;
    case 0x07:    // INC @R1
      fprintf(outFile, "RAM[R1]++;\n");
      break;
    case 0x08:    // INC R0
    case 0x09:    // INC R1
    case 0x0A:    // INC R2
    case 0x0B:    // INC R3
    case 0x0C:    // INC R4
    case 0x0D:    // INC R5
    case 0x0E:    // INC R6
    case 0x0F:    // INC R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "R%01X++;\n", tmp8);
      break;
    case 0x10:    // JBC bit addr, code addr
      fprintf(outFile, "// Warning: JBC Not implemented yet!\n");
      break;
    case 0x13:    // RRC A
      fprintf(outFile, "// Warning: RRC A Not implemented yet!\n");
      break;
    case 0x14:    // DEC A
      fprintf(outFile, "accumulator--;\n");
      break;
    case 0x15:    // DEC data addr
      fprintf(outFile, "RAM[0x%02X]--;\n", data[1]);
      break;
    case 0x16:    // DEC @R0
      fprintf(outFile, "RAM[R0]--;\n");
      break;
    case 0x17:    // DEC @R1
      fprintf(outFile, "RAM[R1]--;\n");
      break;
    case 0x18:    // DEC R0
    case 0x19:    // DEC R1
    case 0x1A:    // DEC R2
    case 0x1B:    // DEC R3
    case 0x1C:    // DEC R4
    case 0x1D:    // DEC R5
    case 0x1E:    // DEC R6
    case 0x1F:    // DEC R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "R%01X--;\n", tmp8);
      break;
    case 0x20:    //JB bit addr, code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[2];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;

      if (data[1] >= 0x80) {
        fprintf(outFile, "if (bitRead(%s, %s%d) == true) goto label_%03X;\n", registers[(data[1] & 0b01111000) >> 3].name, "bit", data[1] & 0b00000111, tmp16 );
      } else {
        fprintf(outFile, "if (bitRead(RAM[0x%02X], %s%d) == true) goto label_%03X;\n", 0x20 + ((data[1] & 0b01111000) >> 3), "bit", data[1] & 0b00000111, tmp16 );
      }
      break;
    case 0x23:    // RL A
      fprintf(outFile, "// Warning: RL A Not implemented yet!\n");
      break;
    case 0x24:    // ADD a, #data
      fprintf(outFile, "// Warning: ADD a, #data, flags not fully implemented\n");
      fprintf(outFile, "tmp16 = (uint16_t)accumulator + 0x%02X;\n", data[1]);
      fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
      fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
      break;
    case 0x25:    // ADD a, data addr
      fprintf(outFile, "// Warning: ADD a, data addr, flags not fully implemented\n");
      fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)RAM[0x%02X];\n", data[1]);
      fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
      fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
      break;
    case 0x26:    // ADD a, @R0
      fprintf(outFile, "// Warning: ADD a, @R0, flags not fully implemented\n");
      fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)RAM[R0];\n");
      fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
      fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
      break;
    case 0x27:    // ADD a, @R1
      fprintf(outFile, "// Warning: ADD a, @R1, flags not fully implemented\n");
      fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)RAM[R1];\n");
      fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
      fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
      break;
    case 0x28:    // ADD a, R0
    case 0x29:    // ADD a, R1
    case 0x2A:    // ADD a, R2
    case 0x2B:    // ADD a, R3
    case 0x2C:    // ADD a, R4
    case 0x2D:    // ADD a, R5
    case 0x2E:    // ADD a, R6
    case 0x2F:    // ADD a, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "// Warning: ADD a, Rn, flags not fully implemented\n");
      fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R%01X;\n", tmp8);
      fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
      fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
      break;
    case 0x30:    // JNB bit addr, code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[2];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;

      if (data[1] >= 0x80) {
        fprintf(outFile, "if (bitRead(%s, %s%d) == false) goto label_%03X;\n", registers[(data[1] & 0b01111000) >> 3].name, "bit", data[1] & 0b00000111, tmp16 );
      } else {
        fprintf(outFile, "if (bitRead(RAM[0x%02X], %s%d) == false) goto label_%03X;\n", 0x20 + ((data[1] & 0b01111000) >> 3), "bit", data[1] & 0b00000111, tmp16 );
      }
      break;

    case 0x33:    // RLC A
      fprintf(outFile, "if (bitRead(accumulator, 7)) tmpCarry = true;\n");
      fprintf(outFile, "accumulator = accumulator << 1;\n");
      fprintf(outFile, "if (tmpCarry) bitSet(accumulator, 0);\n");
      fprintf(outFile, "carry = tmpCarry;\n");
      break;
    case 0x34:    // ADDC A, #data
      fprintf(outFile, "// Warning: ADDC a, #data, flags not fully implemented\n");
      fprintf(outFile, "accumulator += carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator += ");
      fprintf(outFile, "0x%02X;\t// carry affected!\n", data[1]);
      break;
    case 0x35:    // ADDC A, data addr
      fprintf(outFile, "// Warning: ADDC a, data addr, flags not fully implemented\n");
      fprintf(outFile, "accumulator += carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator += ");
      fprintf(outFile, "RAM[0x%02X];\t// carry affected!\n", data[1]);
      break;
    case 0x36:    // ADDC A, @R0
      fprintf(outFile, "// Warning: ADDC a, @R0, flags not fully implemented\n");
      fprintf(outFile, "accumulator += carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator += ");
      fprintf(outFile, "RAM[R0];\t// carry affected!\n");
      break;
    case 0x37:    // ADDC A, @R1
      fprintf(outFile, "// Warning: ADDC a, @R1, flags not fully implemented\n");
      fprintf(outFile, "accumulator += carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator += ");
      fprintf(outFile, "RAM[R1];\t// carry affected!\n");
      break;
    case 0x38:    // ADDC a, R0
    case 0x39:    // ADDC a, R1
    case 0x3A:    // ADDC a, R2
    case 0x3B:    // ADDC a, R3
    case 0x3C:    // ADDC a, R4
    case 0x3D:    // ADDC a, R5
    case 0x3E:    // ADDC a, R6
    case 0x3F:    // ADDC a, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "// Warning: ADDC a, Rr, flags not fully implemented\n");
      fprintf(outFile, "accumulator += carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator += ");
      fprintf(outFile, "R%01X;\t// carry affected!\n", tmp8);
      break;
    case 0x40:    // JC address
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[1];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if ( carry ) goto label_%03X;\n", tmp16 );
      break;
    case 0x42:    // ORL data addr, A
      fprintf(outFile, "RAM[0x%02X] |= accumulator;\n", data[1]);
      break;
    case 0x43:    // ORL data addr, #data
      fprintf(outFile, "RAM[0x%02X] |= 0x%02X;\n", data[1], data[2]);
      break;
    case 0x44:    // ORL A, #data
      fprintf(outFile, "accumulator |= 0x%02X;\n", data[1]);
      break;
    case 0x45:    // ORL A, data addr
      fprintf(outFile, "accumulator |= RAM[0x%02X];\n", data[1]);
      break;
    case 0x46:    // ORL A, @R0
      fprintf(outFile, "accumulator |= RAM[R0];\n");
      break;
    case 0x47:    // ORL A, @R1
      fprintf(outFile, "accumulator |= RAM[R1];\n");
      break;
    case 0x48:    // ORL A, R0
    case 0x49:    // ORL A, R1
    case 0x4A:    // ORL A, R2
    case 0x4B:    // ORL A, R3
    case 0x4C:    // ORL A, R4
    case 0x4D:    // ORL A, R5
    case 0x4E:    // ORL A, R6
    case 0x4F:    // ORL A, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "accumulator |= R%01X;\n", tmp8);
      break;
    case 0x50:    // JNC address
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[1];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if ( carry == false ) goto label_%03X;\n", tmp16);
      break;
    case 0x52:    // ANL data addr, A
      fprintf(outFile, "RAM[0x%02X] &= accumulator;\n", data[1]);
      break;
    case 0x53:    // ANL data addr, #data
      fprintf(outFile, "RAM[0x%02X] &= 0x%02X;\n", data[1], data[2]);
      break;
    case 0x54:    // ANL A, #data
      fprintf(outFile, "accumulator &= 0x%02X;\n", data[1]);
      break;
    case 0x55:    // ANL A, data addr
      fprintf(outFile, "accumulator &= RAM[0x%02X];\n", data[1]);
      break;
    case 0x56:    // ANL A, @R0
      fprintf(outFile, "accumulator &= RAM[R0];\n");
      break;
    case 0x57:    // ANL A, @R1
      fprintf(outFile, "accumulator &= RAM[R1];\n");
      break;
    case 0x58:    // ANL A, R0
    case 0x59:    // ANL A, R1
    case 0x5A:    // ANL A, R2
    case 0x5B:    // ANL A, R3
    case 0x5C:    // ANL A, R4
    case 0x5D:    // ANL A, R5
    case 0x5E:    // ANL A, R6
    case 0x5F:    // ANL A, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "accumulator &= R%01X;\n", tmp8);
      break;
    case 0x60:    // JZ address
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[1];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if ( accumulator == 0 ) goto label_%03X;\n", tmp16);
      break;
    case 0x62:    // XLR data addr, A
      fprintf(outFile, "RAM[0x%02X] ^= accumulator;\n", data[1]);
      break;
    case 0x63:    // XLR data addr, #data
      fprintf(outFile, "RAM[0x%02X] ^= 0x%02X;\n", data[1], data[2]);
      break;
    case 0x64:    // XLR A, #data
      fprintf(outFile, "accumulator ^= 0x%02X;\n", data[1]);
      break;
    case 0x65:    // XLR A, data addr
      fprintf(outFile, "accumulator ^= RAM[0x%02X];\n", data[1]);
      break;
    case 0x66:    // XLR A, @R0
      fprintf(outFile, "accumulator ^= RAM[R0];\n");
      break;
    case 0x67:    // XLR A, @R1
      fprintf(outFile, "accumulator ^= RAM[R1];\n");
      break;
    case 0x68:    // XLR A, R0
    case 0x69:    // XLR A, R1
    case 0x6A:    // XLR A, R2
    case 0x6B:    // XLR A, R3
    case 0x6C:    // XLR A, R4
    case 0x6D:    // XLR A, R5
    case 0x6E:    // XLR A, R6
    case 0x6F:    // XLR A, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "accumulator ^= R%01X];\n", tmp8);
      break;
    case 0x70:    // JNZ address
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[1];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if ( accumulator != 0 ) goto label_%03X;\n", tmp16);
      break;
    case 0x72:    // ORL C, bit addr
      fprintf(outFile, "// Warning: ORL C, bit addr, Not implemented yet!\n");
      break;
    case 0x73:    // JMP @A + DPTR
      fprintf(outFile, "// Warning: // JMP @A + DPTR, Not implemented because runtime stuff!\n");
      break;
    case 0x74:    // MOV A, #data
      fprintf(outFile, "accumulator = 0x%02X;\n", data[1]);
      break;
    case 0x75:    // MOV data addr, #data
      fprintf(outFile, "RAM[0x%02X] = 0x%02X;\n", data[1], data[2]);
      break;
    case 0x76:    // MOV @R0, #data
      fprintf(outFile, "RAM[R0] = 0x%02X;\n", data[1]);
      break;
    case 0x77:    // MOV @R1, #data
      fprintf(outFile, "RAM[R1] = 0x%02X;\n", data[1]);
      break;
    case 0x78:    // MOV R0, #data
    case 0x79:    // MOV R1, #data
    case 0x7A:    // MOV R2, #data
    case 0x7B:    // MOV R3, #data
    case 0x7C:    // MOV R4, #data
    case 0x7D:    // MOV R5, #data
    case 0x7E:    // MOV R6, #data
    case 0x7F:    // MOV R7, #data
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "R%01X = 0x%02X;\n", tmp8, data[1]);
      break;
    case 0x80:    // SJMP code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[1];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "goto label_%03X;\n", tmp16);
      break;
    case 0x82:    // ANL C, bit addr
      fprintf(outFile, "// Warning: ANL C, bit addr, Not implemented yet!\n");
      break;
    case 0x83:    // MOVC A, @A + PC
      fprintf(outFile, "// Warning: // MOVC A, @A + PC, Not implemented because runtime stuff!\n");
      fprintf(outFile, "// accumulator = ROM[accumulator + PC];\n");
      break;
    case 0x84:    // DIV AB
      fprintf(outFile, "// Warning: // DIV AB, Flags not implemented!\n");
      fprintf(outFile, "accumulator = accumulator / regB;\n");
      fprintf(outFile, "regB = accumulator %% regB;\n");
      break;
    case 0x85:    // MOV data addr, data addr
      fprintf(outFile, "RAM[0x%02X] = RAM[0x%02X];\n", data[2], data[1]);
      break;
    case 0x86:    // MOV data addr, @R0
      fprintf(outFile, "RAM[0x%02X] = RAM[R0];\n", data[1]);
      break;
    case 0x87:    // MOV data addr, @R0
      fprintf(outFile, "RAM[0x%02X] = RAM[R1];\n", data[1]);
      break;
    case 0x88:    // MOV data addr, R0
    case 0x89:    // MOV data addr, R1
    case 0x8A:    // MOV data addr, R2
    case 0x8B:    // MOV data addr, R3
    case 0x8C:    // MOV data addr, R4
    case 0x8D:    // MOV data addr, R5
    case 0x8E:    // MOV data addr, R6
    case 0x8F:    // MOV data addr, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "RAM[0x%02X] = R%01X;\n", data[1], tmp8);
      break;
    case 0x90:    // MOV DPTR, #data
      //tmp16 = ((uint16_t)(data[1] & 0b11100000) << 3) + (uint16_t)data[2];
      tmp16 = (((uint16_t)data[1]) << 8) + (uint16_t)data[2];
      fprintf(outFile, "DPTR = 0x%04X;\n", tmp16);
      break;
    case 0x92:    // MOV bit addr, C
      fprintf(outFile, "// Warning: MOV bit addr,C, Not implemented yet!\n");
      break;
    case 0x93:    // MOVC A, @A + DPTR
      fprintf(outFile, "// Warning: // MOVC A, @A + DPTR, Not implemented because runtime stuff!\n");
      fprintf(outFile, "// accumulator = ROM[accumulator + DPTR];\n");
      break;
    case 0x94:    // SUBB A, #data
      fprintf(outFile, "// Warning: SUBB a, #data, flags not fully implemented\n");
      fprintf(outFile, "accumulator -= carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator -= ");
      fprintf(outFile, "0x%02X;\t// carry affected!\n", data[1]);
      break;
    case 0x95:    // SUBB A, data addr
      fprintf(outFile, "// Warning: SUBB a, data addr, flags not fully implemented\n");
      fprintf(outFile, "accumulator -= carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator -= ");
      fprintf(outFile, "RAM[0x%02X];\t// carry affected!\n", data[1]);
      break;
    case 0x96:    // SUBB A, @R0
      fprintf(outFile, "// Warning: SUBB a, @R0, flags not fully implemented\n");
      fprintf(outFile, "accumulator -= carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator -= ");
      fprintf(outFile, "RAM[R0];\t// carry affected!\n");
      break;
    case 0x97:    // SUBB A, @R1
      fprintf(outFile, "// Warning: SUBB a, @R1, flags not fully implemented\n");
      fprintf(outFile, "accumulator -= carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator -= ");
      fprintf(outFile, "RAM[R1];\t// carry affected!\n");
      break;
    case 0x98:    // SUBB a, R0
    case 0x99:    // SUBB a, R1
    case 0x9A:    // SUBB a, R2
    case 0x9B:    // SUBB a, R3
    case 0x9C:    // SUBB a, R4
    case 0x9D:    // SUBB a, R5
    case 0x9E:    // SUBB a, R6
    case 0x9F:    // SUBB a, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "// Warning: SUBB a, Rr, flags not fully implemented\n");
      fprintf(outFile, "accumulator -= carry ? 1 : 0;\t// carry affected!\n");
      fprintf(outFile, "accumulator -= ");
      fprintf(outFile, "R%01X;\t// carry affected!\n", tmp8);
      break;
    case 0xA0:    // ORL C, /bit addr
      fprintf(outFile, "// Warning: ORL C,/bit addr, Not implemented yet!\n");
      break;
    case 0xA2:    // MOV C, bit addr
      if (data[1] >= 0x80) {
        fprintf(outFile, "if (bitRead(%s, %s%d) == true) {\n", registers[(data[1] & 0b01111000) >> 3].name, "bit", data[1] & 0b00000111);
        fprintf(outFile, "\tcarry = true;\n");
        fprintf(outFile, "\t} else {\n");
        fprintf(outFile, "\t\tcarry = false;\n");
        fprintf(outFile, "\t};\n");
      } else {
        fprintf(outFile, "if (bitRead(RAM[0x%02X], %s%d) == true) {\n", 0x20 + ((data[1] & 0b01111000) >> 3), "bit", data[1] & 0b00000111 );
        fprintf(outFile, "\tcarry = true;\n");
        fprintf(outFile, "\t} else {\n");
        fprintf(outFile, "\t\tcarry = false;\n");
        fprintf(outFile, "\t};\n");
      }
      break;
    case 0xA3:    // INC DPTR
      fprintf(outFile, "DPTR++;\n");
      break;
    case 0xA4:    // MUL AB
      fprintf(outFile, "tmp16 = (uint16_t)accumulator * (uint16_t)regB;\n");
      fprintf(outFile, "accumulator = (uint8_t) ( (uint16_t)tmp16 & 0xFF) ;\n");
      fprintf(outFile, "regB = (uint8_t) ( ((uint16_t)tmp16 >> 8) & 0xFF) ;\n");
      break;
    case 0xA5:    // reserved
      fprintf(outFile, "// Warning: reserved opcode\n");
      break;
    case 0xA6:    // MOV @R0, data addr
      fprintf(outFile, "RAM[R0] = RAM[0x%02X];\n", data[1]);
      break;
    case 0xA7:    // MOV @R1, data addr
      fprintf(outFile, "RAM[R1] = RAM[0x%02X];\n", data[1]);
      break;
    case 0xA8:    // MOV R0, data addr
    case 0xA9:    // MOV R1, data addr
    case 0xAA:    // MOV R2, data addr
    case 0xAB:    // MOV R3, data addr
    case 0xAC:    // MOV R4, data addr
    case 0xAD:    // MOV R5, data addr
    case 0xAE:    // MOV R6, data addr
    case 0xAF:    // MOV R7, data addr
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "R%01X = RAM[0x%02X];\n", tmp8, data[1]);
      break;
    case 0xB0:    // ANL C, /bit addr
      fprintf(outFile, "// Warning: ANL C,/bit addr, Not implemented yet!\n");
      break;
    case 0xB2:    // CPL bit addr
      fprintf(outFile, "// Warning: CPL bit addr, Not implemented yet!\n");
      break;
    case 0xB3:    // CPL C
      fprintf(outFile, "!carry; // invert carry\n");
      break;
    case 0xB4:    // CJNE  A, #data, code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[2];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if (accumulator != 0x%02X) goto label_%03X;\n", data[1], tmp16);
      break;
    case 0xB5:    // CJNE  A, data addr, code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[2];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if (accumulator != RAM[0x%02X]) goto label_%03X;\n", data[1], tmp16);
      break;
    case 0xB6:    // CJNE  @R0, #data, code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[2];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if (RAM[R0] != 0x%02X) goto label_%03X;\n", data[1], tmp16);
      break;
    case 0xB7:    // CJNE  @R1, #data, code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[2];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if (RAM[R1] != 0x%02X) goto label_%03X;\n", data[1], tmp16);
      break;
    case 0xB8:    // CJNE  R0, #data, code addr
    case 0xB9:    // CJNE  R1, #data, code addr
    case 0xBA:    // CJNE  R2, #data, code addr
    case 0xBB:    // CJNE  R3, #data, code addr
    case 0xBC:    // CJNE  R4, #data, code addr
    case 0xBD:    // CJNE  R5, #data, code addr
    case 0xBE:    // CJNE  R6, #data, code addr
    case 0xBF:    // CJNE  R7, #data, code addr
      tmp8 = data[0] & 0b00000111;
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[2];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "if (RAM[R%01X] != 0x%02X) goto label_%03X;\n", tmp8, data[1], tmp16);
      break;
    case 0xC0:    // PUSH data addr
      fprintf(outFile, "// Warning: PUSH 0x%02X not implemented Yet!\n", data[1]);
      break;
    case 0xC2:    // CLR bit addr
      if (data[1] >= 0x80) {
        fprintf(outFile, "bitClear(%s, %s%d);\n", registers[(data[1] & 0b01111000) >> 3].name, "bit", data[1] & 0b00000111 );
      } else {
        fprintf(outFile, "bitClear(RAM[0x%02X], %s%d);\n", 0x20 + ((data[1] & 0b01111000) >> 3), "bit", data[1] & 0b00000111 );
      }
      break;
    case 0xC3:    // CLR C
      fprintf(outFile, "carry = false; // clear carry\n");
      break;
    case 0xC4:    // SWAP A
      fprintf(outFile, "tmp8 = (accumulator & 0xF0) >> 4;\n");
      fprintf(outFile, "accumulator = ((accumulator & 0x0F) << 4) | tmp;\n");
      break;
    case 0xC5:    // XCH A, data addr
      fprintf(outFile, "tmp8 = RAM[0x%02X];\n", data[1]);
      fprintf(outFile, "RAM[0x%02X] = accumulator;\n", data[1]);
      fprintf(outFile, "accumulator = tmp8;\n");
      break;
    case 0xC6:    // XCH A, @R0
      fprintf(outFile, "tmp8 = RAM[R0];\n");
      fprintf(outFile, "RAM[R0] = accumulator;\n");
      fprintf(outFile, "accumulator = tmp8;\n");
      break;
    case 0xC7:    // XCH A, @R1
      fprintf(outFile, "tmp8 = RAM[R1];\n");
      fprintf(outFile, "RAM[R1] = accumulator;\n");
      fprintf(outFile, "accumulator = tmp8;\n");
      break;
    case 0xC8:    // XCH A, R0
    case 0xC9:    // XCH A, R1
    case 0xCA:    // XCH A, R2
    case 0xCB:    // XCH A, R3
    case 0xCC:    // XCH A, R4
    case 0xCD:    // XCH A, R5
    case 0xCE:    // XCH A, R6
    case 0xCF:    // XCH A, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "tmp8 = R%01X;\n", tmp8);
      fprintf(outFile, "R%01X = accumulator;\n", tmp8);
      fprintf(outFile, "accumulator = tmp8;\n");
      break;
    case 0xD0:    // POP data addr
      fprintf(outFile, "// Warning: POP 0x%02X not implemented Yet!\n", data[1]);
      break;
    case 0xD2:    // SETB bit addr
      if (data[1] >= 0x80) {
        fprintf(outFile, "bitSet(%s, %s%d);\n", registers[(data[1] & 0b01111000) >> 3].name, "bit", data[1] & 0b00000111 );
      } else {
        fprintf(outFile, "bitSet(RAM[0x%02X], %s%d);\n", 0x20 + ((data[1] & 0b01111000) >> 3), "bit", data[1] & 0b00000111 );
      }
      break;
    case 0xD3:    // SETB C
      fprintf(outFile, "carry = true; // set carry\n");
      break;
    case 0xD4:    // DA A
      fprintf(outFile, "// Warning: DA A, Not implemented yet!\n");
      break;
    case 0xD5:    // DJNZ data addr, code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[2];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      //labels[tmp16] = true;
      fprintf(outFile, "RAM[0x%02X]--;\n", data[1]);
      fprintf(outFile, "if (RAM[0x%02X] == 0) goto label_%03X;\n", data[1], tmp16);
      break;
    case 0xD6:    // XCHD A, @R0
      fprintf(outFile, "tmp = RAM[R0] & 0x0F;\n");
      fprintf(outFile, "RAM[R0] = (RAM[R0] & 0xF0) | (accumulator & 0x0F);\n");
      fprintf(outFile, "accumulator = (accumulator & 0xF0) | (tmp & 0x0F);\n");
      break;
    case 0xD7:    // XCHD A, @R1
      fprintf(outFile, "tmp = RAM[R1] & 0x0F;\n");
      fprintf(outFile, "RAM[R1] = (RAM[R1] & 0xF0) | (accumulator & 0x0F);\n");
      fprintf(outFile, "accumulator = (accumulator & 0xF0) | (tmp & 0x0F);\n");
      break;
    case 0xD8:    // DJNZ R0, code addr
    case 0xD9:    // DJNZ R1, code addr
    case 0xDA:    // DJNZ R2, code addr
    case 0xDB:    // DJNZ R3, code addr
    case 0xDC:    // DJNZ R4, code addr
    case 0xDD:    // DJNZ R5, code addr
    case 0xDE:    // DJNZ R6, code addr
    case 0xDF:    // DJNZ R7, code addr
      tmp16 = progCnt + opcodeLen[data[0]];
      tmp8int = (int8_t)data[1];
      if (tmp8int < 0) {
        tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
      } else {
        tmp16 = tmp16 + (uint16_t)tmp8int;
      }
      tmp8 = data[0] & 0b00000111;
      //labels[tmp16] = true;
      fprintf(outFile, "R%01X--;\n", tmp8);
      fprintf(outFile, "if (R%01X != 0) goto label_%03X;\n", tmp8, tmp16);
      break;
    case 0xE0:    // MOVX A, @DPTR
      fprintf(outFile, "accumulator = extRAM[DPTR];\t// external RAM\n");
      break;
    case 0xE2:    // MOVX A, @R0
      fprintf(outFile, "accumulator = extRAM[R0];\t// external RAM\n");
      break;
    case 0xE3:    // MOVX A, @R1
      fprintf(outFile, "accumulator = extRAM[R1];\t// external RAM\n");
      break;
    case 0xE4:    // CLR A
      fprintf(outFile, "accumulator = 0;\n");
      break;
    case 0xE5:    // MOV A, data addr
      fprintf(outFile, "accumulator = RAM[0x%02X];\n", data[1]);
      break;
    case 0xE6:    // MOV A, @R0
      fprintf(outFile, "accumulator = RAM[R0];\n");
      break;
    case 0xE7:    // MOV A, @R1
      fprintf(outFile, "accumulator = RAM[R1];\n");
      break;
    case 0xE8:    // MOV A, R0
    case 0xE9:    // MOV A, R1
    case 0xEA:    // MOV A, R2
    case 0xEB:    // MOV A, R3
    case 0xEC:    // MOV A, R4
    case 0xED:    // MOV A, R5
    case 0xEE:    // MOV A, R6
    case 0xEF:    // MOV A, R7
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "accumulator = R%01X;\n", tmp8);
      break;
    case 0xF0:    // MOVX @DPTR, A
      fprintf(outFile, "extRAM[DPTR] = accumulator;\t// external RAM\n");
      break;
    case 0xF2:    // MOVX @R0, A
      fprintf(outFile, "extRAM[R0] = accumulator;\t// external RAM\n");
      break;
    case 0xF3:    // MOVX @R1, A
      fprintf(outFile, "extRAM[R1] = accumulator;\t// external RAM\n");
      break;
    case 0xF4:    // CPL A
      fprintf(outFile, "accumulator = ~accumulator;\n");
      break;
    case 0xF5:    // MOV data addr, A
      fprintf(outFile, "RAM[0x%02X] = accumulator;\n", data[1]);
      break;
    case 0xF6:    // MOV @R0, A
      fprintf(outFile, "RAM[R0] = accumulator;\n");
      break;
    case 0xF7:    // MOV @R1, A
      fprintf(outFile, "RAM[R1] = accumulator;\n");
      break;
    case 0xF8:    // MOV R0, A
    case 0xF9:    // MOV R1, A
    case 0xFA:    // MOV R2, A
    case 0xFB:    // MOV R3, A
    case 0xFC:    // MOV R4, A
    case 0xFD:    // MOV R5, A
    case 0xFE:    // MOV R6, A
    case 0xFF:    // MOV R7, A
      tmp8 = data[0] & 0b00000111;
      fprintf(outFile, "R%01X = accumulator;\n", tmp8);
      break;

    default:
      fprintf(outFile, "// Error: something fucked up in decompiler!\n");
      break;
  }
}

// estimate only jump/call label address usage
void partialDecompile(uint8_t data[], uint16_t progCnt) {

  uint16_t tmp16;
  
  switch(data[0]) {
    case 0x01:    // AJMP
    case 0x21:    // AJMP
    case 0x41:    // AJMP
    case 0x61:    // AJMP
    case 0x81:    // AJMP
    case 0xA1:    // AJMP
    case 0xC1:    // AJMP
    case 0xE1:    // AJMP
    case 0x11:    // ACALL
    case 0x31:    // ACALL
    case 0x51:    // ACALL
    case 0x71:    // ACALL
    case 0x91:    // ACALL
    case 0xB1:    // ACALL
    case 0xD1:    // ACALL
    case 0xF1:    // ACALL
      tmp16 = ((uint16_t)(data[0] & 0b11100000) << 3) + (uint16_t)data[1];
      tmp16 = ( (progCnt+1) & 0b1111100000000000) + tmp16;
      labels[tmp16] = true;
      break;
    case 0x02:    // LJMP
    case 0x12:    // LCALL
      tmp16 = ((uint16_t)data[1] << 8) + (uint16_t)data[2];
      labels[tmp16] = true;
      break;
    // 2 byte opcodes, address in data[1]
    case 0x40:    // JC address
    case 0x50:    // JNC address
    case 0x60:    // JZ address
    case 0x70:    // JNZ address
    case 0x80:    // SJMP code addr
    case 0xD8:    // DJNZ R0, code addr
    case 0xD9:    // DJNZ R1, code addr
    case 0xDA:    // DJNZ R2, code addr
    case 0xDB:    // DJNZ R3, code addr
    case 0xDC:    // DJNZ R4, code addr
    case 0xDD:    // DJNZ R5, code addr
    case 0xDE:    // DJNZ R6, code addr
    case 0xDF:    // DJNZ R7, code addr
      tmp16 = relativeAddress(progCnt, data[0], data[1]);
      labels[tmp16] = true;
      break;
    // 3 byte opcodes, address in data[2]
    case 0x20:    // JB bit addr, code addr
    case 0x30:    // JNB bit addr, code addr
    case 0xB4:    // CJNE  A, #data, code addr
    case 0xB5:    // CJNE  A, data addr, code addr
    case 0xB6:    // CJNE  @R0, #data, code addr
    case 0xB7:    // CJNE  @R1, #data, code addr
    case 0xB8:    // CJNE  R0, #data, code addr
    case 0xB9:    // CJNE  R1, #data, code addr
    case 0xBA:    // CJNE  R2, #data, code addr
    case 0xBB:    // CJNE  R3, #data, code addr
    case 0xBC:    // CJNE  R4, #data, code addr
    case 0xBD:    // CJNE  R5, #data, code addr
    case 0xBE:    // CJNE  R6, #data, code addr
    case 0xBF:    // CJNE  R7, #data, code addr
    case 0xD5:    // DJNZ data addr, code addr
      tmp16 = relativeAddress(progCnt, data[0], data[2]);
      labels[tmp16] = true;
      break;
    default:
      break;
  }
}

// to calculate target address relative to program counter (-128 to +127)
uint16_t relativeAddress(uint16_t progCount, uint8_t opcode0, uint8_t relAddr) {
  uint16_t tmp16;
  int8_t tmp8int;
  
  tmp16 = progCount + opcodeLen[opcode0];
  tmp8int = (int8_t)relAddr;
  if (tmp8int < 0) {
    tmp16 = tmp16 - (uint16_t)(abs(tmp8int));
  } else {
    tmp16 = tmp16 + (uint16_t)tmp8int;
  }
  return tmp16;
}
