#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <stdbool.h>

int PC = 0;
bool bank = false;

FILE *inFile;
FILE *outFile;

int decompile(uint8_t data);

int main(int argc, char **argv) {
    inFile = fopen("Compass_KBC.bin", "rb");
    if (inFile == NULL) {
        printf("Error opening file Compass_KBC.bin!\n");
        return 1;
    }

    
    outFile = fopen("Compass_KBC_convert.c", "w");
    if (outFile == NULL) {
        printf("Error opening file Compass_KBC_convert.c!\n");
        return 1;
    }

    fprintf(outFile, "#include <stdio.h>\n");
    fprintf(outFile, "#include <stdlib.h>\n");
    fprintf(outFile, "#include <stdint.h>\n");
    fprintf(outFile, "#include <stdbool.h>\n");
    fprintf(outFile, "int main(int argc, char **argv) {\n");

    while ( true ) {
        uint8_t retVal;
        size_t err;
        
        err = fread(&retVal, 1, 1, inFile);
        if (err != 1) {
            printf("Error reading from source in while loop!\n");
            exit(-1);
        }
        printf("label_%03X:\n", PC);
        if (PC == 0xDB) bank = false;
        if (PC == 0x2CA) bank = true;
        if (PC < 0x300) {
            fprintf(outFile, "label_%03X:\n", PC);
            decompile(retVal);
        } else {
            if (PC == 0x300) {
                fprintf(outFile, "};\n");
                fprintf(outFile, "uint8_t page3Array[] = {\n");
            }
            fprintf(outFile, "0x%02X,\n", retVal);
        }
        PC++;
    }
    
    fclose(inFile);
    fclose(outFile);

    return 0;
}

// RL A (page 55)
int decompile(uint8_t data) {
    uint8_t tmp;
    size_t err;
    
    switch(data) {
        case 0b00000000:    // NOP
            fprintf(outFile, "// NOP\n");
            break;
        case 0b00000001:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b00000010:    // OUT DBB,A
            fprintf(outFile, "busData = accumulator;\n");
            break;
        case 0b00000011:    // ADD A,#data
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + 0x%02X;\n", tmp);
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b00000100:    // JMP address
            fprintf(outFile, "goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", tmp);
            break;
        case 0b00000101:    // EN I
            fprintf(outFile, "extIRQ = true;\n");
            break;
        case 0b00000110:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b00000111:    // DEC A
            fprintf(outFile, "accumulator--;\n");
            break;
        case 0b00001000:    // IN A,Pp
            fprintf(outFile, "accumulator = PORT[0];\t// Invalid port!\n");
            break;
        case 0b00001001:    // IN A,Pp
            fprintf(outFile, "accumulator = PORT[1];\n");
            break;
        case 0b00001010:    // IN A,Pp
            fprintf(outFile, "accumulator = PORT[2];\n");
            break;
        case 0b00001011:    // IN A,Pp
            fprintf(outFile, "accumulator = PORT[3];\t// Invalid port!\n");
            break;
        case 0b00001100:    // MOV A,Pp
            fprintf(outFile, "accumulator = PORT[4];\n");
            break;
        case 0b00001101:    // MOV A,Pp
            fprintf(outFile, "accumulator = PORT[5];\n");
            break;
        case 0b00001110:    // MOV A,Pp
            fprintf(outFile, "accumulator = PORT[6];\n");
            break;
        case 0b00001111:    // MOV A,Pp
            fprintf(outFile, "accumulator = PORT[7];\n");
            break;
        case 0b00010000:    // INC @Rr
            fprintf(outFile, "RAM[R0");
            fprintf(outFile, "%s]++;\n", bank ? "b1" : "b0");
            break;
        case 0b00010001:    //C IN @Rr
            fprintf(outFile, "RAM[R1");
            fprintf(outFile, "%s]++;\n", bank ? "b1" : "b0");
            break;
        case 0b00010010:    // JBb address
            fprintf(outFile, "if ( bitRead(accumulator, 0) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b00010011:    // ADDC A, #data
            fprintf(outFile, "accumulator += carry ? 1 : 0;\t// carry affected!\n");
            fprintf(outFile, "accumulator += ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\t// carry affected!\n", tmp);
            break;
        case 0b00010100:    // CALL address
            fprintf(outFile, "label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X();\t// this is call, check for correct return address\n", (PC & 0xF00) | tmp);
            break;
        case 0b00010101:    // DIS I
            fprintf(outFile, "extIRQ = false;\n");
            break;
        case 0b00010110:    // JTF address
            fprintf(outFile, "if ( timerOvefFlow ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b00010111:    // INC A
            fprintf(outFile, "accumulator++;\n");
            break;
        case 0b00011000:    // INC Rr
            fprintf(outFile, "R0");
            fprintf(outFile, "%s++;\n", bank ? "b1" : "b0");
            break;
        case 0b00011001:    // INC Rr
            fprintf(outFile, "R1");
            fprintf(outFile, "%s++;\n", bank ? "b1" : "b0");
            break;
        case 0b00011010:    // INC Rr
            fprintf(outFile, "R2");
            fprintf(outFile, "%s++;\n", bank ? "b1" : "b0");
            break;
        case 0b00011011:    // INC Rr
            fprintf(outFile, "R3");
            fprintf(outFile, "%s++;\n", bank ? "b1" : "b0");
            break;
        case 0b00011100:    // INC Rr
            fprintf(outFile, "R4");
            fprintf(outFile, "%s++;\n", bank ? "b1" : "b0");
            break;
        case 0b00011101:    // INC Rr
            fprintf(outFile, "R5");
            fprintf(outFile, "%s++;\n", bank ? "b1" : "b0");
            break;
        case 0b00011110:    // INC Rr
            fprintf(outFile, "R6");
            fprintf(outFile, "%s++;\n", bank ? "b1" : "b0");
            break;
        case 0b00011111:    // INC Rr
            fprintf(outFile, "R7");
            fprintf(outFile, "%s++;\n", bank ? "b1" : "b0");
            break;
        case 0b00100000:    // XCH A,@Rr
            fprintf(outFile, "tmp = RAM[R0");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            fprintf(outFile, "RAM[R0");
            fprintf(outFile, "%s] = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00100001:    // XCH A,@Rr
            fprintf(outFile, "tmp = RAM[R1");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            fprintf(outFile, "RAM[R1");
            fprintf(outFile, "%s] = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00100010:    // IN A, DBB
            fprintf(outFile, "accumulator = busIn;\n");
            break;
        case 0b00100011:    // MOV A,#data
            fprintf(outFile, "accumulator = ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b00100100:    // JMP address
            fprintf(outFile, "goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", 0b100000000 + (uint16_t)tmp);
            break;
        case 0b00100101:    // EN TCNTI
            fprintf(outFile, "timerInterrupt = true;\n");
            break;
        case 0b00100110:    // JNT0 address
            fprintf(outFile, "if ( T0 == 0 ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b00100111:    // CLR A
            fprintf(outFile, "accumulator = 0;\n");
            break;
        case 0b00101000:    // XCH a,Rr
            fprintf(outFile, "tmp = R0");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            fprintf(outFile, "R0");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00101001:    // XCH a,Rr
            fprintf(outFile, "tmp = R1");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            fprintf(outFile, "R1");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00101010:    // XCH a,Rr
            fprintf(outFile, "tmp = R2");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            fprintf(outFile, "R2");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00101011:    // XCH a,Rr
            fprintf(outFile, "tmp = R3");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            fprintf(outFile, "R3");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00101100:    // XCH a,Rr
            fprintf(outFile, "tmp = R4");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            fprintf(outFile, "R4");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00101101:    // XCH a,Rr
            fprintf(outFile, "tmp = R5");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            fprintf(outFile, "R5");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00101110:    // XCH A,Rr
            fprintf(outFile, "tmp = R6");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            fprintf(outFile, "R6");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00101111:    // XCH A,Rr
            fprintf(outFile, "tmp = R7");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            fprintf(outFile, "R7");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = tmp;\n");
            break;
        case 0b00110000:    // XCHD A,@Rr
            fprintf(outFile, "tmp = RAM[R0");
            fprintf(outFile, "%s] & 0x0F;\n", bank ? "b1" : "b0");
            
            fprintf(outFile, "RAM[R0");
            fprintf(outFile, "%s] = (RAM[R0", bank ? "b1" : "b0");
            fprintf(outFile, "%s] & 0xF0) | (accumulator & 0x0F);\n", bank ? "b1" : "b0");
            
            fprintf(outFile, "accumulator = (accumulator & 0xF0) | (tmp & 0x0F);\n");            
            
            break;
        case 0b00110001:    // XCHD A,@Rr
            fprintf(outFile, "tmp = RAM[R1");
            fprintf(outFile, "%s] & 0x0F;\n", bank ? "b1" : "b0");
            
            fprintf(outFile, "RAM[R1");
            fprintf(outFile, "%s] = (RAM[R1", bank ? "b1" : "b0");
            fprintf(outFile, "%s] & 0xF0) | (accumulator & 0x0F);\n", bank ? "b1" : "b0");
            
            fprintf(outFile, "accumulator = (accumulator & 0xF0) | (tmp & 0x0F);\n");            
            
            break;
        case 0b00110010:    // JBb address
            fprintf(outFile, "if ( bitRead(accumulator, 1) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b00110011:
            break;
        case 0b00110100:    // CALL address
            fprintf(outFile, "label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X();\t// this is call, check for correct return address\n", 0b100000000 + (uint16_t)tmp);
            break;
        case 0b00110101:    // DIS TCNT
            fprintf(outFile, "timerInterrupt = false;\n");
            break;
        case 0b00110110:    // JT0 address
            fprintf(outFile, "if ( T0 ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b00110111:    // CPL A
            fprintf(outFile, "accumulator = ~accumulator;\n");
            break;
        case 0b00111000:    // OUTL Pp,A
            fprintf(outFile, "PORT[0] = accumulator;\t// Invalid port!\n");
            break;
        case 0b00111001:    // OUTL Pp,A
            fprintf(outFile, "PORT[1] = accumulator;\n");
            break;
        case 0b00111010:    // OUTL Pp,A
            fprintf(outFile, "PORT[2] = accumulator;\n");
            break;
        case 0b00111011:    // OUTL Pp,A
            fprintf(outFile, "PORT[3] = accumulator;\t// Invalid port!\n");
            break;
        case 0b00111100:    // MOV Pp, A
            fprintf(outFile, "PORT[4] = accumulator;\n");
            break;
        case 0b00111101:    // MOV Pp, A
            fprintf(outFile, "PORT[5] = accumulator;\n");
            break;
        case 0b00111110:    // MOV Pp, A
            fprintf(outFile, "PORT[6] = accumulator;\n");
            break;
        case 0b00111111:    // MOV Pp, A
            fprintf(outFile, "PORT[7] = accumulator;\n");
            break;
        case 0b01000000:    // ORL A,@Rr
            fprintf(outFile, "accumulator |= RAM[R0");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            break;
        case 0b01000001:    // ORL A,@Rr
            fprintf(outFile, "accumulator |= RAM[R1");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            break;
        case 0b01000010:    // MOV A,T
            fprintf(outFile, "accumulator = timer;\n");
            break;
        case 0b01000011:    // ORL A,#data
            fprintf(outFile, "accumulator |=");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b01000100:    // JMP address
            fprintf(outFile, "goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", 0b1000000000 + (uint16_t)tmp);
            break;
        case 0b01000101:    // STRT CNT
            fprintf(outFile, "timerMode = modeCounter;\n");
            fprintf(outFile, "timerEnable = true;\n");
            break;
        case 0b01000110:    // JNT1 address
            fprintf(outFile, "if ( T1 == 0 ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b01000111:    // SWAP A
            fprintf(outFile, "tmp8 = (accumulator & 0xF0) >> 4;\n");
            fprintf(outFile, "accumulator = ((accumulator & 0x0F) << 4) | tmp;\n");
            break;
        case 0b01001000:    // ORL A,Rr
            fprintf(outFile, "accumulator |= R0");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01001001:    // ORL A,Rr
            fprintf(outFile, "accumulator |= R1");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01001010:    // ORL A,Rr
            fprintf(outFile, "accumulator |= R2");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01001011:    // ORL A,Rr
            fprintf(outFile, "accumulator |= R3");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01001100:    // ORL A,Rr
            fprintf(outFile, "accumulator |= R4");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01001101:    // ORL A,Rr
            fprintf(outFile, "accumulator |= R5");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01001110:    // ORL A,Rr
            fprintf(outFile, "accumulator |= R6");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01001111:    // ORL A,Rr
            fprintf(outFile, "accumulator |= R7");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01010000:    // ANL A,@Rr
            fprintf(outFile, "accumulator &= RAM[R0");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            break;
        case 0b01010001:    // ANL A,@Rr
            fprintf(outFile, "accumulator &= RAM[R1");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            break;
        case 0b01010010:    // JBb address
            fprintf(outFile, "if ( bitRead(accumulator, 2) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b01010011:    // ANL A, #data
            fprintf(outFile, "accumulator &= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b01010100:    // CALL address
            fprintf(outFile, "label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X();\t// this is call, check for correct return address\n", 0b1000000000 + (uint16_t)tmp);
            break;
        case 0b01010101:    // STRT T
            fprintf(outFile, "timerMode = modeTimer;\n");
            fprintf(outFile, "timerEnable = true;\n");
            break;
        case 0b01010110:    // JT1 address
            fprintf(outFile, "if ( T1 ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b01010111:    // DA A
            fprintf(outFile, "// decimal adjust A\t// not implemented\n");
            break;
        case 0b01011000:    // ANL A,Rr
            fprintf(outFile, "accumulator &= R0");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01011001:    // ANL A,Rr
            fprintf(outFile, "accumulator &= R1");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01011010:    // ANL A,Rr
            fprintf(outFile, "accumulator &= R2");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01011011:    // ANL A,Rr
            fprintf(outFile, "accumulator &= R3");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01011100:    // ANL A,Rr
            fprintf(outFile, "accumulator &= R4");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01011101:    // ANL A,Rr
            fprintf(outFile, "accumulator &= R5");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01011110:    // ANL A, Rr
            fprintf(outFile, "accumulator &= R6");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01011111:    // ANL A,Rr
            fprintf(outFile, "accumulator &= R7");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b01100000:    // ADD a,@Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)RAM[R0");
            fprintf(outFile, "%s];\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01100001:    // ADD a,@Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)RAM[R1");
            fprintf(outFile, "%s];\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01100010:    // MOV T,A
            fprintf(outFile, "timer = accumulator;\n");
            break;
        case 0b01100011:
            break;
        case 0b01100100:    // JMP address
            fprintf(outFile, "goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", 0b1100000000 + (uint16_t)tmp);
            break;
        case 0b01100101:    // STOP TCNT
            fprintf(outFile, "timerEnabled = false;\n");
            break;
        case 0b01100110:
            break;
        case 0b01100111:    // RRC A
            fprintf(outFile, "if (bitRead(accumulator, 0)) tmpCarry = true;\n");
            fprintf(outFile, "accumulator = accumulator >> 1;\n");
            fprintf(outFile, "if (tmpCarry) bitSet(accumulator, 7);\n");
            fprintf(outFile, "carry = tmpCarry;\n");
            break;
        case 0b01101000:    // ADD a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R0");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01101001:    // ADD a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R1");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01101010:    // ADD a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R2");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01101011:    // ADD a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R3");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01101100:    // ADD a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R4");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01101101:    // ADD a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R5");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01101110:    // ADD a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R6");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01101111:    // ADD a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (uint16_t)R7");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01110000:    // ADDC a,@Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)RAM[R0");
            fprintf(outFile, "%s];\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01110001:    // ADDC a,@Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)RAM[R1");
            fprintf(outFile, "%s];\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01110010:    // JBb address
            fprintf(outFile, "if ( bitRead(accumulator, 3) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b01110011:
            break;
        case 0b01110100:    // CALL address
            fprintf(outFile, "label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X();\t// this is call, check for correct return address\n", 0b110000000 + (uint16_t)tmp);
            break;
        case 0b01110101:
            break;
        case 0b01110110:    // JF1 address
            fprintf(outFile, "if ( bitRead(busStatus, F1) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b01110111:    // RR A
            fprintf(outFile, "accumulator = accumulator >> 1;\n");
            break;
        case 0b01111000:    // ADDC a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)R0");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01111001:    // ADDC a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)R1");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01111010:    // ADDC a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)R2");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01111011:    // ADDC a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)R3");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01111100:    // ADDC a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)R4");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01111101:    // ADDC a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)R5");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01111110:    // ADDC a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)R6");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b01111111:    // ADDC a,Rr
            fprintf(outFile, "tmp16 = (uint16_t)accumulator + (carry ? 1 : 0) + (uint16_t)R7");
            fprintf(outFile, "%s;\t// carry affected!\n", bank ? "b1" : "b0");
            fprintf(outFile, "accumulator = (uint8_t)(tmp16 & 0xFF);\n");
            fprintf(outFile, "if (tmp16 > 0xFF) carry = true;\n");
            break;
        case 0b10000000:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b10000001:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b10000010:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b10000011:    // RET
            fprintf(outFile, "return;\t// Return from subroutine. Check for correct call label!\n");
            break;
        case 0b10000100:    // JMP address
            fprintf(outFile, "goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", 0b10000000000 + (uint16_t)tmp);
            break;
        case 0b10000101:    // CLR F0
            fprintf(outFile, "bitClear(busStatus, F0);\n");
            break;
        case 0b10000110:    // JOBF address
            fprintf(outFile, "if ( bitRead(busStatus, OBF) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b10000111:
            break;
        case 0b10001000:    // ORL Pr,#data
            fprintf(outFile, "PORT[0] |= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\t// Incorrect port!\n", tmp);
            break;
        case 0b10001001:    // ORL Pr,#data
            fprintf(outFile, "PORT[1] |= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10001010:    // ORL Pr,#data
            fprintf(outFile, "PORT[2] |= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10001011:    // ORL Pr,#data
            fprintf(outFile, "PORT[3] |= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\t// Incorrect port!\n", tmp);
            break;
        case 0b10001100:    // ORL Pp,A
            fprintf(outFile, "PORT[4] |= accumulator\n");
            break;
        case 0b10001101:    // ORL Pp,A
            fprintf(outFile, "PORT[5] |= accumulator\n");
            break;
        case 0b10001110:    // ORL Pp,A
            fprintf(outFile, "PORT[6] | = accumulator\n");
            break;
        case 0b10001111:    // ORL Pp,A
            fprintf(outFile, "PORT[7] |= accumulator\n");
            break;
        case 0b10010000:    // MOV STS,A
            fprintf(outFile, "busStatus = (busStatus & 0x0F) | (accumulator & 0xF0);\n");
            break;
        case 0b10010001:
            break;
        case 0b10010010:    // JBb address
            fprintf(outFile, "if ( bitRead(accumulator, 4) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b10010011:    // RETI
            fprintf(outFile, "return;\t// Return from Interrupt\n");
            break;
        case 0b10010100:
            break;
        case 0b10010101:    // CPL F0
            fprintf(outFile, "bitTogle(busStatus, F0);\n");
            break;
        case 0b10010110:    // JNZ address
            fprintf(outFile, "if ( accumulator ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b10010111:    // CLR C
            fprintf(outFile, "carry = false;\n");
            break;
        case 0b10011000:    // ANL Pp,#data
            fprintf(outFile, "PORT[0] &= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\t// This port is invalid!\n", tmp);
            break;
        case 0b10011001:    // ANL Pp,#data
            fprintf(outFile, "PORT[1] &= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10011010:    // ANL Pp,#data
            fprintf(outFile, "PORT[2] &= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10011011:    // ANL Pp,#data
            fprintf(outFile, "PORT[3] &= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\t// This port is invalid!\n", tmp);
            break;
        case 0b10011100:    //ANLD Pp,A
            fprintf(outFile, "PORT[4] &= (accumulator & 0x0F);\n");
            break;
        case 0b10011101:    //ANLD Pp,A
            fprintf(outFile, "PORT[5] &= (accumulator & 0x0F);\n");
            break;
        case 0b10011110:    //ANLD Pp,A
            fprintf(outFile, "PORT[6] &= (accumulator & 0x0F);\n");
            break;
        case 0b10011111:    //ANLD Pp,A
            fprintf(outFile, "PORT[7] &= (accumulator & 0x0F);\n");
            break;
        case 0b10100000:    // MOV @Rr,A
            fprintf(outFile, "RAM[R0");
            fprintf(outFile, "%s] = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10100001:    // MOV @Rr,A
            fprintf(outFile, "RAM[R1");
            fprintf(outFile, "%s] = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10100010:
            break;
        case 0b10100011:    // MOV A,@A
            fprintf(outFile, "accumulator = ROM[ (PC &0xF00) | accumulator];\t// Not implemented yet\n");
            break;
        case 0b10100100:    // JMP address
            fprintf(outFile, "goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", 0b10100000000 + (uint16_t)tmp);
            break;
        case 0b10100101:    // CLR F1
            fprintf(outFile, "bitClear(busStatus, F1);\n");
            break;
        case 0b10100110:
            break;
        case 0b10100111:    // CPL C
            fprintf(outFile, "carry = !carry;\n");
            break;
        case 0b10101000:    // MOV Rr,A
            fprintf(outFile, "R0");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10101001:    // MOV Rr,A
            fprintf(outFile, "R1");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10101010:    // MOV Rr,A
            fprintf(outFile, "R2");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10101011:    // MOV Rr,A
            fprintf(outFile, "R3");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10101100:    // MOV Rr,A
            fprintf(outFile, "R4");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10101101:    // MOV Rr,A
            fprintf(outFile, "R5");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10101110:    // MOV Rr,A
            fprintf(outFile, "R6");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10101111:    // MOV Rr,A
            fprintf(outFile, "R7");
            fprintf(outFile, "%s = accumulator;\n", bank ? "b1" : "b0");
            break;
        case 0b10110000:    // MOV @Rr,#data
            fprintf(outFile, "RAM[R0");
            fprintf(outFile, "%s] = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10110001:    // MOV @Rr,#data
            fprintf(outFile, "RAM[R1");
            fprintf(outFile, "%s] = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10110010:    // JBb address
            fprintf(outFile, "if ( bitRead(accumulator, 5) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b10110011:    // JMPP @A
            fprintf(outFile, "// goto label_[(PC & 0xF00) | accumulator];\t// This is runtime stuff! Need some fixing!\n");
            break;
        case 0b10110100:
            break;
        case 0b10110101:    // CPL F1
            fprintf(outFile, "bitTogle(busStatus, F1);\n");
            break;
        case 0b10110110:    // JF0 address
            fprintf(outFile, "if ( bitRead(busStatus, F0) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b10110111:
            break;
        case 0b10111000:    // MOV Rr,#data
            fprintf(outFile, "R0");
            fprintf(outFile, "%s = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10111001:    // MOV Rr,#data
            fprintf(outFile, "R1");
            fprintf(outFile, "%s = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10111010:    // MOV Rr,#data
            fprintf(outFile, "R2");
            fprintf(outFile, "%s = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10111011:    // MOV Rr,#data
            fprintf(outFile, "R3");
            fprintf(outFile, "%s = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10111100:    // MOV Rr,#data
            fprintf(outFile, "R4");
            fprintf(outFile, "%s = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10111101:    // MOV Rr,#data
            fprintf(outFile, "R5");
            fprintf(outFile, "%s = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10111110:    // MOV Rr,#data
            fprintf(outFile, "R6");
            fprintf(outFile, "%s = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b10111111:    // MOV Rr,#data
            fprintf(outFile, "R7");
            fprintf(outFile, "%s = ", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);
            break;
        case 0b11000000:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11000001:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11000010:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11000011:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11000100:    // JMP address
            fprintf(outFile, "goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", 0b11000000000 + (uint16_t)tmp);
            break;
        case 0b11000101:    // SEL RB0
            fprintf(outFile, "// Bank0 in use\n");
            bank = false;
            break;
        case 0b11000110:    // JZ address
            fprintf(outFile, "if ( accumulator == 0 ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11000111:    // MOV A, PSW
            fprintf(outFile, "accumulator = PSW;\n");
            break;
        case 0b11001000:    // DEC Rr
            fprintf(outFile, "R0");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            break;
        case 0b11001001:    // DEC Rr
            fprintf(outFile, "R1");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            break;
        case 0b11001010:    // DEC Rr
            fprintf(outFile, "R2");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            break;
        case 0b11001011:    // DEC Rr
            fprintf(outFile, "R3");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            break;
        case 0b11001100:    // DEC Rr
            fprintf(outFile, "R4");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            break;
        case 0b11001101:    // DEC Rr
            fprintf(outFile, "R5");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            break;
        case 0b11001110:    // DEC Rr
            fprintf(outFile, "R6");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            break;
        case 0b11001111:    // DEC Rr
            fprintf(outFile, "R7");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            break;
        case 0b11010000:    // XLR A,@Rr
            fprintf(outFile, "accumulator ^= RAM[R0");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            break;
        case 0b11010001:    // XLR A,@Rr
            fprintf(outFile, "accumulator ^= RAM[R1");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            break;
        case 0b11010010:    // JBb address
            fprintf(outFile, "if ( bitRead(accumulator, 6) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11010011:    // XLR A,#data
            fprintf(outFile, "accumulator ^= ");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "0x%02X;\n", tmp);

            break;
        case 0b11010100:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11010101:    // SEL RB1
            fprintf(outFile, "// Bank1 in use\n");
            bank = true;
            break;
        case 0b11010110:    // JNIBF address
            fprintf(outFile, "if ( bitRead(busStatus, IBF) == 0 ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11010111:    // MOV PSW,A
            fprintf(outFile, "PSW = accumulator;\n");
            break;
        case 0b11011000:    // XLR A,Rr
            fprintf(outFile, "accumulator ^= R0");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11011001:    // XLR A,Rr
            fprintf(outFile, "accumulator ^= R1");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11011010:    // XLR A,Rr
            fprintf(outFile, "accumulator ^= R2");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11011011:    // XLR A,Rr
            fprintf(outFile, "accumulator ^= R3");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11011100:    // XLR A,Rr
            fprintf(outFile, "accumulator ^= R4");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11011101:    // XLR A,Rr
            fprintf(outFile, "accumulator ^= R5");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11011110:    // XLR A,Rr
            fprintf(outFile, "accumulator ^= R6");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11011111:    // XLR A,Rr
            fprintf(outFile, "accumulator ^= R7");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11100000:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11100001:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11100010:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11100011:    //MOVP3 A,@A
            fprintf(outFile, "accumulator = page3Array[accumulator];\t// Double check for correct array name!\n");
            break;
        case 0b11100100:    // JMP address
            fprintf(outFile, "goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", 0b11100000000 + (uint16_t)tmp);
            break;
        case 0b11100101:    // En DMA
            fprintf(outFile, "modeDMA = true;\n");
            break;
        case 0b11100110:    // JNC address
            fprintf(outFile, "if ( carry == false ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11100111:    // RL A
            fprintf(outFile, "accumulator = accumulator << 1;\n");
            break;
        case 0b11101000:    // DJNZ Rr
            fprintf(outFile, "R0");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            fprintf(outFile, "if (R0");
            fprintf(outFile, "%s != 0) goto label_", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11101001:    // DJNZ Rr
            fprintf(outFile, "R1");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            fprintf(outFile, "if (R1");
            fprintf(outFile, "%s == 0) goto label_", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11101010:    // DJNZ Rr
            fprintf(outFile, "R2");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            fprintf(outFile, "if (R2");
            fprintf(outFile, "%s != 0) goto label_", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11101011:    // DJNZ Rr
            fprintf(outFile, "R3");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            fprintf(outFile, "if (R3");
            fprintf(outFile, "%s != 0) goto label_", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11101100:    // DJNZ Rr
            fprintf(outFile, "R4");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            fprintf(outFile, "if (R4");
            fprintf(outFile, "%s != 0) goto label_", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11101101:    // DJNZ Rr
            fprintf(outFile, "R5");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            fprintf(outFile, "if (R5");
            fprintf(outFile, "%s != 0) goto label_", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11101110:    // DJNZ Rr
            fprintf(outFile, "R6");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            fprintf(outFile, "if (R6");
            fprintf(outFile, "%s != 0) goto label_", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11101111:    // DJNZ Rr
            fprintf(outFile, "R7");
            fprintf(outFile, "%s--;\n", bank ? "b1" : "b0");
            fprintf(outFile, "if (R7");
            fprintf(outFile, "%s != 0) goto label_", bank ? "b1" : "b0");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11110000:    // MOV A,@Rr
            fprintf(outFile, "accumulator = RAM[R0");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            break;
        case 0b11110001:    // MOV A,@Rr
            fprintf(outFile, "accumulator = RAM[R1");
            fprintf(outFile, "%s];\n", bank ? "b1" : "b0");
            break;
        case 0b11110010:    // JBb address
            fprintf(outFile, "if ( bitRead(accumulator, 7) ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11110011:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11110100:
            fprintf(outFile, "// Illegal opcode 0x%02X\n", data);
            break;
        case 0b11110101:    // EN FLAGS
            fprintf(outFile, "globalIRQ = true;\n");
            break;
        case 0b11110110:    // JC address
            fprintf(outFile, "if ( carry ) goto label_");
            err = fread(&tmp, 1, 1, inFile);
            if (err != 1) {
                printf("Error reading from source while in switch case!\n");
                exit(-1);
            }
            PC++;
            fprintf(outFile, "%03X;\n", (PC & 0xF00) | tmp);
            break;
        case 0b11110111:    // RLC A
            fprintf(outFile, "if (bitRead(accumulator, 7)) tmpCarry = true;\n");
            fprintf(outFile, "accumulator = accumulator << 1;\n");
            fprintf(outFile, "if (tmpCarry) bitSet(accumulator, 0);\n");
            fprintf(outFile, "carry = tmpCarry;\n");
            break;
        case 0b11111000:    // MOV A, Rr
            fprintf(outFile, "accumulator = R0");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11111001:    // MOV A, Rr
            fprintf(outFile, "accumulator = R1");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11111010:    // MOV A, Rr
            fprintf(outFile, "accumulator = R2");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11111011:    // MOV A, Rr
            fprintf(outFile, "accumulator = R3");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11111100:    // MOV A, Rr
            fprintf(outFile, "accumulator = R4");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11111101:    // MOV A, Rr
            fprintf(outFile, "accumulator = R5");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11111110:    // MOV A, Rr
            fprintf(outFile, "accumulator = R6");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
        case 0b11111111:    // MOV A, Rr
            fprintf(outFile, "accumulator = R7");
            fprintf(outFile, "%s;\n", bank ? "b1" : "b0");
            break;
    }

    return 0;
}
